%This script calculates the melting loop for FePt alloys and compares it to
%experimental data.

%This script was used to calcualte Fig. 4 in 
% Hirschmann, M.M. and Zhang, H.L. (2023) 
% A revised model for activity-composition relations in solid and molten 
% FePt alloys and a preliminary high pressure model for characterization 
% of oxygen fugacity in high pressure experiments.
% European Journal of Mineralogy

%Original Script
%16 April 2023, MMH
%Commented and cleaned up 16 August 2023, MMH

%Script needs to be able to read file SGTEIronPlatinum.xlsx


R=8.314;






% n is the composition counter, nMax will be the number of composition steps

%Asymmetric Solution Parameters

disp('Choose solid alloy model.  Kessel or new model')

%"New Model = Hirschmann and Zhang, European Journal of Mineralogy, 2023

prompt="1=Kessel, 2=new model\n";
fccmodel = input(prompt);
if(fccmodel==1)
    fccstr='Kessel ';
    W12fcc= -138000;
    W21fcc= -90800;
    W12liq= -140800;
    W21liq= -93200;
else
    fccstr='new model ';
    W12fcc= -121500;
    W21fcc= -91300;
    W12liq= -124500;
    W21liq= -94000;
end

disp('high resolution or low resolution?')
disp('If low resolution is selected, the phase diagram should appear in a moment')
disp('If high resolution is selected, be VERY patient....')
disp('maybe go have a cup of coffee while you wait')

prompt="3=high resolution, 4= low resolution\n";
HiLo=input(prompt);
if HiLo==3
    Tstep=0.5;
    nMax=4001;

else
    Tstep=5;
    nMax=201;
end

    %Tn will be the temperature counter, 
    % TnMax is the number of temperature steps
Tmax=2200;
Tmin=1700;

TnMax=1+(Tmax-Tmin)/Tstep;

Tplot=zeros(TnMax:1);
for ii=1:TnMax
Tplot(ii)=Tmin+(ii-1)*Tstep;
end


xFe=zeros(nMax:1);
Phase=zeros(TnMax,nMax,'int8');

%Phase=1 is fcc, Phase=2 is liq, Phase 3 is fcc+liq
Gfcc=zeros(TnMax,nMax);
Gliq=zeros(TnMax,nMax);
Ginterceptfcc=zeros(TnMax,nMax);
Ginterceptliq=zeros(TnMax,nMax);
Gslopefcc=zeros(TnMax,nMax);
Gslopeliq=zeros(TnMax,nMax);
GfccFe=zeros(TnMax,1);
GfccPt=zeros(TnMax,1);
GliqFe=zeros(TnMax,1);
GliqPt=zeros(TnMax,1);

xFe(1)=0.000001;
xFe(nMax)=0.999999;
for i=2:nMax-1
    xFe(i)=(i-1)/(nMax-1);
end

TableInput =readtable('SGTEIronPlatinum.xlsx','Sheet','Data','ReadVariableNames',false);

S = vartype('numeric');
Gparams= TableInput{:,S};
N=width(Gparams);

THiLo=Gparams(1,:);
aLo=Gparams(2,:);
bLo=Gparams(3,:);
cLo=Gparams(4,:);
dLo=Gparams(5,:);
eLo=Gparams(6,:);
fLo=Gparams(7,:);
gLo=Gparams(8,:);
hLo=Gparams(9,:);
aHi=Gparams(10,:);
bHi=Gparams(11,:);
cHi=Gparams(12,:);
dHi=Gparams(13,:);
eHi=Gparams(14,:);
fHi=Gparams(15,:);
gHi=Gparams(16,:);
hHi=Gparams(17,:);
Tmag=Gparams(18,:);
B0=Gparams(19,:);
D=Gparams(20,:);

 %   H=zeros(Tsteps+1,N+1);

for Tn=1:TnMax
   
   T=Tmin+(Tn-1)*Tstep;
   G=zeros(N,1);
 
%This loop calculates G (standard state)

for gg= 1:N
    if T>THiLo(gg)
    G(gg)=aHi(gg) + bHi(gg)*T + cHi(gg)*T*log(T) + dHi(gg)*T^2 + eHi(gg)*T^3 + fHi(gg)/T + gHi(gg)*T^7 + hHi(gg)*T^-9;
   % H(gg+1)=-(6*gHi(gg)*T^16 + 2*eHi(gg)*T^12 + dHi(gg)*T^11 + cHi(gg)*T^10 - aHi(gg)*T^9 - 2*fHi(gg)*T^8 - 10*hHi(gg))/T^9;
    else
    G(gg)=aLo(gg) + bLo(gg)*T + cLo(gg)*T*log(T) + dLo(gg)*T^2 + eLo(gg)*T^3 + fLo(gg)/T + gLo(gg)*T^7 + hLo(gg)*T^-9;
   %H(Tloop+1,gg+1)=-(6*gLo(gg)*T^16 + 2*eLo(gg)*T^12 + dLo(gg)*T^11 + cLo(gg)*T^10 - aLo(gg)*T^9 - 2*fLo(gg)*T^8 - 10*hLo(gg))/T^9;
    end

    if(Tmag(gg)>0) 
            G(gg)=G(gg)-R*T*log(B0(gg))*(((T/Tmag(gg))^-5/10+(T/Tmag(gg))^-15/315+(T/Tmag(gg))^-25/1500)/D(gg));
         %   H(Tloop+1,gg+1)=H(Tloop+1,gg+1)-R*T*log(B0(gg)+1)*(-(T/Tmag(gg))^-5/2+(T/Tmag(gg))^-15/21+(T/Tmag(gg))^-25/60)/D(gg);
            
    end
end

GfccFe(Tn)=G(1);
GliqFe(Tn)=G(4);


%Not presently using SGTE Pt values.  it is out of date and has a
%discontinuity near Tfusion

%Values instead are from Arblaster (2005)

GfccPt(Tn)=120.891*T-3.949695e-3*T^2+6.2915e-7*T^3-1.278758e-10*T^4+13848.75/T-23.8992*T*log(T)-7539.23;

GliqPt(Tn)=227.57*T-38.9928*T*log(T)-3805.63;

end

%Now generate G-x curves for fcc and liq phases at each temperature

for Tn=1:TnMax
    for n=1:nMax

        T=Tmin+(Tn-1)*5;

Gfcc(Tn,n)=xFe(n)*GfccFe(Tn)+(1-xFe(n))*GfccPt(Tn)...
    +R*T*(xFe(n)*log(xFe(n))+(1-xFe(n))*log(1-xFe(n)))...
    +(W12fcc*(1-xFe(n))+W21fcc*xFe(n))*xFe(n)*(1-xFe(n));
Gliq(Tn,n)=xFe(n)*GliqFe(Tn)+(1-xFe(n))*GliqPt(Tn)...
    +R*T*(xFe(n)*log(xFe(n))+(1-xFe(n))*log(1-xFe(n)))...
    +(W12liq*(1-xFe(n))+W21liq*xFe(n))*xFe(n)*(1-xFe(n));
if (n>1 && n<nMax)
Gslopefcc(Tn,n)=((Gfcc(Tn,n)-Gfcc(Tn,n-1))/(xFe(n)-xFe(n-1)));  
Gslopeliq(Tn,n)=(Gliq(Tn,n)-Gliq(Tn,n-1))/(xFe(n)-xFe(n-1)) ;
end
   if n==1
       Gslopefcc(Tn,n)=(Gfcc(Tn,n+1)-Gfcc(Tn,n))/(xFe(n+1)-xFe(n));
       Gslopeliq(Tn,n)=(Gliq(Tn,n+1)-Gliq(Tn,n))/(xFe(n+1)-xFe(n));
   end

   if n==nMax
       Gslopefcc(Tn,n)=(Gfcc(Tn,n)-Gfcc(Tn,n-1))/(xFe(n)-xFe(n-1)); 
       Gslopeliq(Tn,n)=(Gliq(Tn,n)-Gliq(Tn,n-1))/(xFe(n)-xFe(n-1));
   end

   Ginterceptfcc(Tn,n)=Gfcc(Tn,n)-Gslopefcc(Tn,n)*xFe(n);
   Ginterceptliq(Tn,n)=Gliq(Tn,n)-Gslopeliq(Tn,n)*xFe(n);

   %If Gfcc>Gliq, provisionally assign state point Phase(T, XFe) to liquid
   %(=3)
   %If Gfcc>Gliq, provisionally assign state point Phase(T, XFe) to solid
   %(=1)


          if Gfcc(Tn,n)>Gliq(Tn,n)
             Phase(Tn,n)=3;
         else
             Phase(Tn,n)=1;
         end
    
%Loop through composition space to see if tangent line to phase is ever
%above curve for complementary phase.  If so, state point Phase(T,XFe) is
%in two phase region (=2).

    for j=1:nMax

        Test1=Gfcc(Tn,j)-Ginterceptliq(Tn,n)-xFe(j)*Gslopeliq(Tn,n);
        Test2=Gliq(Tn,j)-Ginterceptfcc(Tn,n)-xFe(j)*Gslopefcc(Tn,n);
        if (Phase(Tn,n)==3 && Test1<0)
            Phase(Tn,n)=2;
        end
        if (Phase(Tn,n)==1 && Test2<0)
            Phase(Tn,n)=2;
        end

     end
             

    end
end

plottemp=10;
plottangentlocation=20;

tangentline=zeros(nMax);
for k=1:nMax
    tangentline(k)=Ginterceptfcc(plottemp,plottangentlocation)+Gslopefcc(plottemp,plottangentlocation)*xFe(k);
end

%Plot experimental data from Fredriksson (2004), Isaac and Tammann (1907)
%and Buckley and Hume-Rothery (1959)
F04x=[0.4808 0.703 0.9048];
F04y=[1875 1826.6 1789.2];
IT07y=[1800 1798 1795 1780 1773 1797 1800 1814 1819 1832 1877 1929 2042];
IT07x=[1 0.9927 0.9852 0.9692 0.9332 0.8907 0.8397 0.777 0.7 0.5995 0.4662 0.2796 0];
IT07Soly=[1785 1789 1784 1785 1809 1839];
IT07Solx=[0.8397 0.7774 0.6996 0.5995 0.466 0.2796];
B59x=[1 0.993 0.988 0.9821 0.9797 0.9752 0.9719 0.9718 0.968 0.964 0.9573 0.9548 0.9458 0.94 0.929 0.929 0.9025];
B59y=[1804.4 1799.5 1793.7 1795.5 1791.2 1791.9 1788.3 1790.4 1790 1790 1787.1 1788.9 1790.3 1791 1790.2 1790 1791.8];
B59Soly=[1796 1794 1793.5 1794.4 1794.5 1791.9 1791.6 1793.5 1793 1793.4 1793.3 1796.4 1795.5 1798 1799.1 1801.5 1805.7];
B59Solx=[0.9025 0.929 0.929 0.94 0.9458 0.9548 0.9573 0.964 0.968 0.9718 0.9719 0.9752 0.9797 0.9821 0.988 0.993 1];



ContourLevels=2:3;

colorsBRG = [0 0 1
             1 0 0
             0 1 0];
         
colororder(colorsBRG);
set(gca,'nextplot','replacechildren')
set(gca,'colororder',colorsBRG)

figure
[M]=contour(xFe,Tplot,Phase,ContourLevels,'-');
hold on
plot(F04x,F04y,'o')
plot(IT07x,IT07y,'^')
plot(IT07Solx,IT07Soly,'square')
plot(B59x,B59y,'x')
plot(B59Solx,B59Soly,'+')
%  If value in "contour" command is [2 2], then add text in title 'solidus
%  countour'
title([fccstr 'W12, W21 liquid: ' num2str(W12liq) ', ' num2str(W21liq)])
xlabel('XFe/(XFe+XPt)') 
ylabel('T(K)') 
legend({'fit','F04','IT07','ITsol','B59','B59sol'},'Location','southwest')
ax = gca;
ax.FontSize = 18;


d=size(M);

outputarray3=zeros(2,nMax);


begin=d(2)-nMax+1;

for jj=begin:nMax
    outputarray3(1,jj)=M(1,jj);
    outputarray3(2,jj)=M(2,jj);
end


fid3=fopen('ContourData.txt','w');
Table2=array2table(outputarray3);
writetable(Table2,'ContourData.txt');
fclose(fid3);



